/* Yeasu GS-232/G5400B Rotor Controller
Replaces Yeasu GS-232A
Written by Glen Popiel, KW5GP


 This program is free software: you can redistribute it and/or modify
 it under the terms of the version 3 GNU General Public License as
 published by the Free Software Foundation.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 
 */

#define debug_mode 0  // Set to 1 for debug data on Serial Port

#include <Wire.h>  // Include the I2C Communication Library
#include <EEPROM.h>  // Include the EEPROM Library
#include "ADS1115.h" // Include the ADS1115.h Library (Library Updated to fix errors)
#include "I2Cdev.h"  // Include I2CDev.h Library (Library Updated to fix errors)

ADS1115 adc; // Define the ADS1115 as adc

#define rotate_up 3  // Define Rotate Up as Pin 3
#define rotate_down 2  // Define Rotate Down as Pin 2
#define rotate_left 4  // Define Rotate Left as Pin 4
#define rotate_right 5  // Define Rotate Right as Pin 5

#define BAUD_RATE 9600  // Set the Serial Port Baud rate to 9600

#define EEPROM_ID_BYTE 1   // EEPROM ID to validate EEPROM data location
#define EEPROM_ID  55  // EEPROM ID Value
#define EEPROM_AZ_CAL_0 2    // Azimuth Zero Calibration EEPROM location     
#define EEPROM_AZ_CAL_MAX 4  // Azimuth Max Calibration Data EEPROM location  
#define EEPROM_EL_CAL_0 6    // Elevation Zero Calibration Data EEPROM location  
#define EEPROM_EL_CAL_MAX 8  // Elevation Max Calibration Data EEPROM location  

#define AZ_CAL_0_DEFAULT 0        // Preset the Azimuth Zero Calibration Point to 0
#define AZ_CAL_MAX_DEFAULT 27000        // Preset the Azimuth Max Calibration Point to 27000
#define EL_CAL_0_DEFAULT 0        // Preset the Elevation Zero Calibration Point to 0
#define EL_CAL_MAX_DEFAULT 27000        // Preset the Elevation Max Calibration Point to 27000

#define AZ_Tolerance 0  // Set the Azimuth Accuracy Tolerance
#define EL_Tolerance 0  // Set the Elevation Accuracy Toleance

//variables
byte inByte = 0;  // incoming serial byte
byte serial_buffer[50];  // incoming serial byte buffer
int serial_buffer_index = 0;  // The index pointer variable for the Serial buffer
int set_AZ;  // Azimuth set value          
int set_EL;  // Elevation set value
int current_AZ;  // Current Azimuth raw value
int current_EL;  // Current Elevation raw value
String Serial_Send_Data; // Data to send to Serial Port
int AZ_0;  // Azimuth Zero Value from EEPROM
int AZ_MAX; // Azimuth Max Value from EEPROM
int EL_0;  // Elevation 0 Value from EEPROM
int EL_MAX;  //Elevation Max Value from EEPROM
int AZ_Degrees; // mapped AZ ADC value to Degrees
int EL_Degrees; // mapped EL ADC value to Degrees
String Requested_AZ; // RS232 Requested Azimuth - M and short W command
String Requested_EL; //RS232 Requested Azimuth and Elevation - Full W command
int AZ_To; // Requested AZ Move 
int EL_To; // Requested EL Move
int AZ_Distance; // Distance to move AZ
int EL_Distance; // Distance to move EL

void setup() 
{
  pinMode(rotate_up, OUTPUT);  // Define the Control Pins as Outputs
  pinMode(rotate_down, OUTPUT);
  pinMode(rotate_left, OUTPUT);
  pinMode(rotate_right, OUTPUT);
  
  digitalWrite(rotate_up, LOW);  // Turn off all the Control Pins
  digitalWrite(rotate_down, LOW);
  digitalWrite(rotate_left, LOW);
  digitalWrite(rotate_right, LOW);
  
  Serial.begin(BAUD_RATE); // initialize serial communication 

  Wire.begin();  // join I2C bus
  
  adc.initialize(); // initialize ADS1115 16 bit A/D chip
  
  Wire.beginTransmission(0x48); // Begin direct ADC communication
  Wire.write(0x1);    //  Connect to adc and send two bytes - Set Config Reg to all Ones
  Wire.write(0x7F);  //  MSB 
  Wire.write(0xFF);    //  LSB
  Wire.endTransmission();  // End the direct ADC Communication
  
  adc.setMode(ADS1115_MODE_CONTINUOUS); // Set the ADC to free running conversion mode
  adc.setGain(ADS1115_PGA_6P144);  // set the ADC gain to 6.144 Volt range, .0001875 Volts/step
  adc.setRate(ADS1115_RATE_475);  // set ADC sample rate to 475 samples per second
  adc.setMultiplexer(ADS1115_MUX_P0_NG);  // Set the ADC to AN0+ Vs ground Mode

  set_AZ = -1;  // Preset the Azimuth and Elevation Move Variables
  set_EL = -1;
  
  read_eeprom_cal_data();  // Read the Azimuth and Elevation Calibration Values from EEPROM
  
}  // End Setup Loop

void loop() 
{
  check_serial();  // Check the Serial Port for Data
  check_move();  // Check to see if executing move command
} // End Main Loop

// Functions

void read_eeprom_cal_data()  // Function to Read the Azimuth and Elevation Calibration Data
{
  if (EEPROM.read(EEPROM_ID_BYTE) == EEPROM_ID) // Verify the EEPROM has valid data
  {
    if (debug_mode) // If in Debug Mode Print the Calibration Values
    {
      Serial.println("Read EEPROM Calibration Data Valid ID");
      Serial.println((EEPROM.read(EEPROM_AZ_CAL_0) * 256) + EEPROM.read(EEPROM_AZ_CAL_0 + 1),DEC);      
      Serial.println((EEPROM.read(EEPROM_AZ_CAL_MAX) * 256) + EEPROM.read(EEPROM_AZ_CAL_MAX + 1),DEC);      
      Serial.println((EEPROM.read(EEPROM_EL_CAL_0) * 256) + EEPROM.read(EEPROM_EL_CAL_0 +1 ),DEC);
      Serial.println((EEPROM.read(EEPROM_EL_CAL_MAX) *256) + EEPROM.read(EEPROM_EL_CAL_MAX + 1),DEC);
    }
    
    AZ_0 = (EEPROM.read(EEPROM_AZ_CAL_0)*256) + EEPROM.read(EEPROM_AZ_CAL_0 + 1);  // Read the Azimuth Zero Calibration Value from EEPROM
    AZ_MAX = (EEPROM.read(EEPROM_AZ_CAL_MAX)*256) + EEPROM.read(EEPROM_AZ_CAL_MAX + 1);  // Read the Azimuth Maximum Calibration Value from EEPROM
    EL_0 = (EEPROM.read(EEPROM_EL_CAL_0)*256) + EEPROM.read(EEPROM_EL_CAL_0 + 1);  // Read the Elevation Zero Calibration Value from EEPROM
    EL_MAX = (EEPROM.read(EEPROM_EL_CAL_MAX)*256) + EEPROM.read(EEPROM_EL_CAL_MAX + 1);  // Read the Elevation Maximum Calibration Value from EEPROM
    
  } else {  // initialize eeprom to default values
  if (debug_mode) 
  {
    Serial.println("Read EEPROM Calibration Data Invalid ID - setting to defaults");
  }
  AZ_0 = AZ_CAL_0_DEFAULT;  // Set the Calibration To Default Values
  AZ_MAX = AZ_CAL_MAX_DEFAULT;
  EL_0 = EL_CAL_0_DEFAULT;
  EL_MAX = EL_CAL_MAX_DEFAULT;
  write_eeprom_cal_data();  // Write the Default Values to EEPROM
  }
}

void write_eeprom_cal_data() // Function to Write the Calibration Values to EEPROM
{
  if (debug_mode) 
  {
    Serial.println("Writing EEPROM Calibration Data");
  }
  
  EEPROM.write(EEPROM_ID_BYTE,EEPROM_ID);  //   Write the EEPROM ID
  EEPROM.write(EEPROM_AZ_CAL_0,highByte(AZ_0));  // Write the Azimuth Zero Calibration High Order Byte
  EEPROM.write(EEPROM_AZ_CAL_0 + 1,lowByte(AZ_0));    // Write the Azimuth Zero Calibration Low Order Byte
  EEPROM.write(EEPROM_AZ_CAL_MAX,highByte(AZ_MAX));  // Write the Azimuth Max Calibration High Order Byte
  EEPROM.write(EEPROM_AZ_CAL_MAX + 1,lowByte(AZ_MAX));  // Write the Azimuth Max Calibration Low Order Byte
  EEPROM.write(EEPROM_EL_CAL_0,highByte(EL_0));  // Write the Elevation Zero Calibration High Order Byte
  EEPROM.write(EEPROM_EL_CAL_0 + 1,lowByte(EL_0));  // Write the Elevation Zero Calibration Low Order Byte
  EEPROM.write(EEPROM_EL_CAL_MAX,highByte(EL_MAX));  // Write the Elevation Max Calibration High Order Byte
  EEPROM.write(EEPROM_EL_CAL_MAX + 1,lowByte(EL_MAX));  // Write the Elevation Max Calibration Low Order Byte
}

void check_serial() // Function to check for data on the Serial port
{
  if (Serial.available() > 0) // Get the Serial Data if available
  {
    inByte = Serial.read();  // Get the Serial Data
    
    // You may need to comment out the following line if your PC software
    // will not communicate properly with the controller
    // SatPC32 wants the command echoed, Ham Radio Deluxe does not
    Serial.print(char(inByte));  // Echo back to the PC

    if (inByte == 10)  // ignore Line Feeds
    {
    return;
    }
    if (inByte !=13)  // Add to buffer if not CR
    {
      serial_buffer[serial_buffer_index] = inByte;
             
      if (debug_mode) // Print the Character received if in Debug mode
      {
        Serial.print("Received = ");
        Serial.println(serial_buffer[serial_buffer_index]);
      }
      
      serial_buffer_index++;  // Increment the Serial Buffer pointer
    
    } else {  // It's a Carriage Return, execute command

    if ((serial_buffer[0] > 96) && (serial_buffer[0] < 123))  //If first character of command is lowercase, convert to uppercase
    {
      serial_buffer[0] = serial_buffer[0] - 32;
    }
    
    switch (serial_buffer[0]) {  // Decode first character of command
                 
      case 65:  // A Command - Stop the Azimuth Rotation
      
        if (debug_mode) {Serial.println("A Command Received");}
        az_rotate_stop();
        break;
      
      case 66:  // B Command - Send the Current Elevation to the PC
      
        if (debug_mode) {Serial.println("B Command Received");}
        send_current_el();  // Call the Send Current Elevation Function
        break;
      
      case 67:      // C - return current azimuth
      
        if (debug_mode)   // Return the Buffer Index Pointer in Debug Mode
        {
          Serial.println("C Command Received");
          Serial.println(serial_buffer_index);
        }
        if ((serial_buffer_index == 2) & (serial_buffer[1] == 50)) 
        {
          if (debug_mode) 
          {
            Serial.println("C2 Command Received");
          }        
          send_current_azel();  // Return Azimuth and Elevation if C2 Command
        } else { 
          send_current_az();  // Return Azimuth if C Command
        }
        break;
      
      case 68:  // D - Rotate Elevation Down Command
      
        if (debug_mode) 
        {
          Serial.println("D Command Received");
        }
        rotate_el_down();  // Call the Rotate Elevation Down Function
        break;
      
      case 69:  // E - Stop the Elevation Rotation
      
        if (debug_mode) 
        {
          Serial.println("E Command Received");
        }
        el_rotate_stop();  // Call the Elevation Rotation Stop Function
        break;
              
      case 70:  // F - Set the Max Calibration
      
        if (debug_mode) 
        {
          Serial.println("F Command Received");
          Serial.println(serial_buffer_index);
        }
        if ((serial_buffer_index == 2) & (serial_buffer[1] == 50)) // Check for F2 Command
        {
          if (debug_mode) 
          {
            Serial.println("F2 Command Received");
          }        
          set_max_el_cal();  // F2 - Set the Max Elevation Calibration
        } else { 
          set_max_az_cal();  // F - Set the Max Azimuth Calibration
        }      
      break;
      
      case 76:  // L - Rotate Azimuth CCW
      
        if (debug_mode) 
        {
          Serial.println("L Command Received");
        }
        rotate_az_ccw();  // Call the Rotate Azimuth CCW Function
        break;
      
      case 77:  // M - Rotate to Set Point
      
        if (debug_mode) 
        {
          Serial.println("M Command Received");
        }
        rotate_to();  // Call the Rotate to Set Point Command
        break;
            
      case 79:  // O - Set Zero Calibration
      
        if (debug_mode) 
        {
          Serial.println("O Command Received");
          Serial.println(serial_buffer_index);
        }
        if ((serial_buffer_index == 2) & (serial_buffer[1] == 50)) // Check for O2 Command
        {
          if (debug_mode) 
          {
            Serial.println("O2 Command Received");
          }        
          set_0_el_cal();  // O2 - Set the Elevation Zero Calibration
        } else { 
          set_0_az_cal();  // O - Set the Azimuth Zero Calibration
        }       
        break;
      
      case 82:  // R - Rotate Azimuth CW
      
        if (debug_mode) 
        {
          Serial.println("R Command Received");
        }
        rotate_az_cw();  // Call the Rotate Azimuth CW Function
        break;
      
      case 83:  // S - Stop All Rotation
      
        if (debug_mode) 
        {
          Serial.println("S Command Received");
        }
        az_rotate_stop();  // Call the Stop Azimith Rotation Function
        el_rotate_stop();  // Call the Stop Elevation Rotation Function
        break;
      
      case 85:  // U - Rotate Elevation Up
      
        if (debug_mode) 
        {
          Serial.println("U Command Received");
        }
        rotate_el_up();  // Call the Rotate Elevation Up Function
        break;
            
      case 87:  // W - Rotate Azimuth and Elevation to Set Point
      
        if (debug_mode) 
        {
          Serial.println("W Command Received");
        }
        rotate_az_el_to();  // Call the Rotate Azimuth and Elevation to Set Point Function
        break;
      
      }
      
      serial_buffer_index = 0;  // Clear the Serial Buffer and Reset the Buffer Index Pointer
      serial_buffer[0] = 0;
    }
  }
}
  
void send_current_az() // Send the Current Azimuth Function
{
  read_adc();  // Read the ADC
  
  // Map Azimuth to degrees
  if (debug_mode) 
  {
    Serial.println(current_AZ);
  }
  AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 360);  // Map the Current Azimuth to Degrees
  if (AZ_Degrees > 180) // Correction Since Azimuth Reading starts at Meter Center Point
  {
    AZ_Degrees = AZ_Degrees - 180;
  } else { 
    AZ_Degrees = AZ_Degrees + 180;
  }
  if (debug_mode) 
  {
    Serial.println(AZ_Degrees);
  }
  // Send it back via serial
  Serial_Send_Data = "";
  if (AZ_Degrees < 100)  // pad with 0's if needed
  {
    Serial_Send_Data = "0";
  }
  if (AZ_Degrees < 10) 
  {
    Serial_Send_Data = "00";
  }
  Serial_Send_Data = "+0" + Serial_Send_Data + String(AZ_Degrees);  // Send the Azimuth in Degrees
  Serial.println(Serial_Send_Data);  // Return value via RS-232 port
}
  
void send_current_azel() // Function to Send the Current Azimuth and Elevation
{
  read_adc();  // Read the ADC
  
  // Map Azimuth to degrees
  if (debug_mode) 
  {
    Serial.println(current_AZ);
  }
  AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 360);  // Map the Current Azimuth to Degrees
  if (AZ_Degrees > 180) // Correction Since Azimuth Reading starts at Meter Center Point
  {
    AZ_Degrees = AZ_Degrees - 180;
  } else { 
    AZ_Degrees = AZ_Degrees + 180;
  }
   
  // Map Elevation to degrees
  if (debug_mode) 
  {
    Serial.println(current_EL);
  }
  EL_Degrees = map(current_EL, EL_0, EL_MAX, 0, 180);  // Map the Elevation to Degrees
  if (debug_mode) 
  {
    Serial.println(EL_Degrees);
    Serial.println(AZ_Degrees);
  }  
  // Send it back via serial
  Serial_Send_Data = "";
  if (AZ_Degrees < 100)  // pad with 0's if needed
  {
    Serial_Send_Data = "0";
  }
  if (AZ_Degrees < 10) 
  {
    Serial_Send_Data = "00";
  }
  Serial_Send_Data = "+0" + Serial_Send_Data + String(AZ_Degrees) + "+0";  // Send the Azimuth part of the string
  if (EL_Degrees < 100)  // pad with 0's if needed
  {
    Serial_Send_Data = Serial_Send_Data + "0";
  }
  if (EL_Degrees < 10) 
  {
    Serial_Send_Data = Serial_Send_Data+ "0";
  }
  Serial_Send_Data = Serial_Send_Data + String(EL_Degrees);  // Send the Elevation Part of the String
  Serial.println(Serial_Send_Data);  // Return value via RS-232 port  
}

void set_max_az_cal() // Set the Max Azimuth Calibration Function
{  
  if (debug_mode) 
  {
    Serial.println("Calibrate Max AZ Function");
  }    
  read_adc();  // Read the ADC
  
  // save current az and el values to EEPROM - Zero Calibration
  if (debug_mode) 
  {
    Serial.println(current_AZ);
  } 
  AZ_MAX = current_AZ;  // Set the Azimuth Maximum Calibration to Current Azimuth Reading
  write_eeprom_cal_data();  // Write the Calibration Data to EEPROM
  if (debug_mode)
  {
    Serial.println("Max Azimuth Calibration Complete");
  }
}  

void set_max_el_cal()   // Set the Max Elevation Calibration Function
{
  if (debug_mode) 
  {
    Serial.println("Calibrate EL Max Function");
  }    
  read_adc();  // Read the ADC
  
  // save current Azimuth and Elevation values to EEPROM - Zero Calibration
  if (debug_mode) 
  {
    Serial.println(current_EL);
  } 
  EL_MAX = current_EL;  // Set the Elevation Max Calibration to the Current Elevation Reading
  write_eeprom_cal_data();  // Write the Calibration Data to EEPROM
  if (debug_mode)
  {
    Serial.println("Max Elevation Calibration Complete");
  }
}

void rotate_az_ccw() // Function to Rotate Azimuth CCW
{
  digitalWrite(rotate_left, HIGH);  // Set the Rotate Left Pin High
  digitalWrite(rotate_right, LOW);  // Make sure the Rotate Right Pin is Low
}

void rotate_az_cw() // Function to Rotate Azimuth CW
{
  digitalWrite(rotate_right, HIGH);    // Set the Rotate Right Pin High
  digitalWrite(rotate_left, LOW);    // Make sure the Rotate Left Pin Low
}

void rotate_el_up() // Function to Rotate Elevation Up
{
  digitalWrite(rotate_up, HIGH);  // Set the Rotate Up Pin High
  digitalWrite(rotate_down, LOW);  // Make sure the Rotate Down Pin is Low
}

void rotate_el_down()  // Function to Rotate Elevation Up
{
  digitalWrite(rotate_down, HIGH);   // Set the Rotate Down Pin High
  digitalWrite(rotate_up, LOW);    // Make sure the Rotate Up Pin is Low
}

void az_rotate_stop() // Function to Stop Azimuth Rotation
{
  digitalWrite(rotate_right, LOW);  // Turn off the Rotate Right Pin
  digitalWrite(rotate_left, LOW);   // Turn off the Rotate Left Pin
}

void el_rotate_stop() // Function to Stop Elevation Rotation
{
  digitalWrite(rotate_up, LOW); // Turn off the Rotate Up Pin
  digitalWrite(rotate_down, LOW);   // Turn off the Rotate Down Pin
}

void rotate_to() // Function to Rotate to Set Point
{
  if (debug_mode) 
  {
    Serial.println("M Command -  Rotate Azimuth To Function");
  }  
  // Decode Command - Format Mxxx - xxx = Degrees to Move to
  if (debug_mode) 
  {
    Serial.println(serial_buffer_index);
  }
  if (serial_buffer_index == 4)  // Verify the Command is the proper length
  {
    if (debug_mode) 
    {
      Serial.println("Value in [1] to [3]?");
    }
    Requested_AZ = (String(char(serial_buffer[1])) + String(char(serial_buffer[2])) + String(char(serial_buffer[3]))) ;  // Decode the Azimuth Value
    AZ_To = (Requested_AZ.toInt()); // AZ Degrees to Move to as integer
    if (AZ_To <0) // Make sure we don't go below 0 degrees
    {
      AZ_To = 0;
    }
    if (AZ_To >360) // Make sure we don't go over 360 degrees
    {
      AZ_To = 360;
    }
    if (AZ_To > 180) // Adjust for Meter starting at Center
    {
      AZ_To = AZ_To - 180;
    } else { 
      AZ_To = AZ_To + 180;
    }
    if (debug_mode) 
    {
      Serial.println(Requested_AZ);  
      Serial.println(AZ_To);
    }   

    // set the move flag and start
    read_adc();  // Read the ADC

    // Map it to degrees
    if (debug_mode) 
    {
      Serial.println(current_AZ);
    }
    AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 360);  // Map the Azimuth Value to Degrees
    if (debug_mode) 
    {
      Serial.println(AZ_Degrees);
    }
    AZ_Distance = AZ_To - AZ_Degrees;  // Figure out far we have to move
    set_AZ = AZ_To;
    if (abs(AZ_Distance) <= AZ_Tolerance)  // No move needed if we're within the Tolerance Range
    {
      az_rotate_stop();  // Stop the Azimuth Rotation
      set_AZ = -1;  // Turn off the Move Command
    } else {  // Move Azimuth - figure out which way
      if (AZ_Distance > 0)   //We need to move CCW
      {
        rotate_az_ccw();  // If the distance is positive, move CCW
      } else {
        rotate_az_cw();  // Otherwise, move clockwise
      }
    }  
  }  
}

void send_current_el() // Function to Send the Current Elevation
{
  read_adc();  // Read the ADC
  // Map it to degrees
  if (debug_mode) 
  {
    Serial.println(current_EL);
  }
  EL_Degrees = map(current_EL, EL_0, EL_MAX, 0, 180);  // Map the Elevation Value to Degrees
  if (debug_mode) 
  {
    Serial.println(EL_Degrees);
    Serial.println(AZ_Degrees);
  }
  // Send it back via serial
  Serial_Send_Data = "";
  if (EL_Degrees < 100)  // pad with 0's if needed
  {
    Serial_Send_Data = "0";
  }
  if (EL_Degrees < 10) 
  {
    Serial_Send_Data = "00";
  }
  Serial_Send_Data = "+0" + Serial_Send_Data + String(EL_Degrees);  // Send the Elevation String
  Serial.println(Serial_Send_Data);  // Return value via RS-232 port  
}

void rotate_az_el_to() // Rotate Azimuth and Elevation to Set Point Function
{
  if (debug_mode) 
  {
    Serial.println("W Command -  Rotate Azimuth and Elevation To Function");
  }  
  // Decode Command - Format Wxxx yyy where xxx = Azimuth to Move to yyy = Elevation to Move to
  if (debug_mode) 
  {
    Serial.println(serial_buffer_index);
  }
  if (serial_buffer_index == 8)  // Verify the command is the proper length
  {
    if (debug_mode) 
    {
      Serial.println("Value in [1] to [3]?");
    }
    Requested_AZ = (String(char(serial_buffer[1])) + String(char(serial_buffer[2])) + String(char(serial_buffer[3]))) ;  // Decode the Azimuth portion of the command
    AZ_To = (Requested_AZ.toInt()); // AZ Degrees to Move to as integer
    if (AZ_To <0) // Don't allow moving below zero
    {
      AZ_To = 0;
    }
    if (AZ_To >360) // Don't allow moving above 360
    {
      AZ_To = 360;
    }
    if (AZ_To >180) // Adjust for Azimuth starting at Center
    {
      AZ_To = AZ_To - 180;
    } else {
      AZ_To = AZ_To + 180;
    }
    
    if (debug_mode) 
    {
      Serial.println(Requested_AZ);
      Serial.println(AZ_To);
      Serial.println("Value in [5] to [7]?");
    }
    Requested_EL = (String(char(serial_buffer[5])) + String(char(serial_buffer[6])) + String(char(serial_buffer[7]))) ;  // Decode the Elevation portion of the command
    EL_To = (Requested_EL.toInt()); // EL Degrees to Move to as integer
    if (EL_To <0) // Don't allow moving below zero
    {
      EL_To = 0;
    }
    if (EL_To >180) // Don't allow moving above 180
    {
      EL_To = 180;
    }
    if (debug_mode) 
    {
      Serial.println(Requested_EL);  
      Serial.println(EL_To);
    }   
  
    // set the move flag and start
    read_adc();  // Read the ADC
    // Map it to degrees
    if (debug_mode) 
    {
      Serial.println(current_AZ);
    }
    AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 360);  // Map the Azimuth Value to Degrees
    if (debug_mode) 
    {
      Serial.println(AZ_Degrees);
    }
    AZ_Distance = AZ_To - AZ_Degrees;  // Figure how far to move Azimuth
    set_AZ = AZ_To;
    EL_Degrees = map(current_EL, EL_0, EL_MAX, 0, 180);  // Map the Elevation Value to Degrees
    if (debug_mode) 
    {
      Serial.println(EL_Degrees);
    }
    EL_Distance = EL_To - EL_Degrees;  // Figure how far to move Elevation
    set_EL = EL_To;
    // Set Azimuth
    if (abs(AZ_Distance) <= AZ_Tolerance)  // No move needed if we're within tolerance range
    {
      az_rotate_stop();  // Stop the Azimuth Rotation
      set_AZ = -1;  // Turn off the Azimuth Move Command
    } else {  // Move Azimuth - figure out which way
      if (AZ_Distance > 0)   //We need to move CW
      {
        rotate_az_cw();  // Rotate CW if positive
      } else {
        rotate_az_ccw(); // Rotate CCW if negative
      }
    } 
    // Set Elevation
    if (abs(EL_Distance) <= EL_Tolerance)  // No move needed if we're within tolerance range
    {
      el_rotate_stop();  // Stop the Elevation Rotation
      set_EL = -1;  // Turn off the Elevation Move Command
    } else {  // Move Elevation - figure out which way
      if (EL_Distance > 0)   //We need to move CW
      {
        rotate_el_up();  // Rotate Up if positive
      } else {
        rotate_el_down();  // Rotate Down if negative
      }
    }    
  }
}

void set_0_az_cal() // Set Azimuth Zero Calibration
{
  if (debug_mode) 
  {
    Serial.println("Calibrate Zero Function");
  }    
  
  read_adc();  // Read the ADC
  // save current az and el values to EEPROM - Zero Calibration
  if (debug_mode) 
  {
    Serial.println(current_EL);
    Serial.println(current_AZ);
  } 
  AZ_0 = current_AZ;  // Set the Azimuth Zero Calibration to current position
  write_eeprom_cal_data();  // Write the Calibration Data to EEPROM
  if (debug_mode)
  {
    Serial.println("Zero Azimuth Calibration Complete");
  }
}

void set_0_el_cal() //  Set the Elevation Zero Calibration
{
  if (debug_mode) 
  {
    Serial.println("Calibrate Zero Function");
  }
  
  read_adc();  // Read the ADC
  // save current az and el values to EEPROM - Zero Calibration
  if (debug_mode) 
  {
    Serial.println(current_EL);
    Serial.println(current_AZ);
  } 
  EL_0 = current_EL;  // Set the Elevation Zero Calibration to current position
  write_eeprom_cal_data();  // Write the Calibration Data to EEPROM
  if (debug_mode)
  {
    Serial.println("Zero Elevation Calibration Complete");
  }
}

void read_adc() // Function to read the ADC
{
 if (debug_mode) 
 {
   Serial.println("Read ADC Function  ");
 }

  int RotorValue;  // Variable to store the rotor value
  adc.setRate(ADS1115_RATE_475); // Set the ADC rate to 475 samples/sec
  adc.setGain(ADS1115_PGA_6P144); // Set the ADC gain to 6.144V
  adc.setMultiplexer(ADS1115_MUX_P0_NG);  // Set the ADC to Channel 0 AN0+ Vs ground
  delay(10); // adc settling delay
  current_EL = adc.getDiff0();  // Read ADC Channel 0
  if (debug_mode) 
  {
    Serial.println(current_EL);
  }  
  adc.setMultiplexer(ADS1115_MUX_P1_NG);  // Set the ADC to Channel 1 AN1+ Vs ground
  delay(10); // adc settling delay
  current_AZ = adc.getDiff1();  // Read ADC Channel 1
  if (debug_mode) 
  {
    Serial.println(current_AZ);
  }  
}

void check_move() // Check to see if we've been commanded to move
{
  if (set_AZ != -1 || set_EL != -1) {   // We're moving - check and stop as needed
    read_adc();  // Read the ADC
    // Map AZ to degrees
    if (debug_mode) 
    {
      Serial.println(current_AZ);
    }
    AZ_Degrees = map(current_AZ, AZ_0, AZ_MAX, 0, 360);  // Map the Current Azimuth reading to Degrees
    // Map EL to degrees
    if (debug_mode) 
    {
      Serial.println(current_EL);
    }
    EL_Degrees = map(current_EL, EL_0, EL_MAX, 0, 180);  // Map the Current Elevation to Degrees
    if (debug_mode) 
    {
      Serial.println(EL_Degrees); 
      Serial.println(AZ_Degrees);
    }  

    if (set_AZ != -1) // If Azimuth is moving
    {
      AZ_Distance = set_AZ - AZ_Degrees;  // Check how far we have to move
      if (abs(AZ_Distance) <= AZ_Tolerance)  // No move needed if we're within the tolerance range
      {
        az_rotate_stop();  // Stop the Azimuth Rotation
        set_AZ = -1;  // Turn off the Azimuth Move Command
      } else {  // Move Azimuth - figure out which way
        if (AZ_Distance > 0)   //We need to move CW
        {
          rotate_az_cw();  // Rotate CW if positive
        } else {
          rotate_az_ccw();  // Rotate CCW if negative
        }    
      } 
   }
   if (set_EL != -1) // If Elevation is moving
   {
     EL_Distance = set_EL - EL_Degrees;  // Check how far we have to move
     if (abs(EL_Distance) <= EL_Tolerance) { // No move needed if we're within tolerance range
       el_rotate_stop();  // Stop the Elevation Rotation
       set_EL = -1;  // Turn off the Elevation Move Command
      } else {  // Move Azimuth - figure out which way
        if (EL_Distance > 0)   //We need to move CW
        {
          rotate_el_up();  // Rotate Up if positive
        } else {
          rotate_el_down();  // Rotate Down if negative
        }    
      }   
    }
  } 
}

